using System;
using System.IO;
using System.Collections;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Web;
using System.Web.SessionState;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;

using System.Runtime.Serialization;
using System.Reflection;

using Atalasoft.Imaging;
using Atalasoft.Imaging.Codec;
using Atalasoft.Imaging.Codec.Tiff;
using Atalasoft.Imaging.ImageProcessing;
using Atalasoft.Imaging.WebControls;
using Atalasoft.Annotate;
using Atalasoft.Annotate.UI;
using Atalasoft.Imaging.ImageProcessing.Transforms;

namespace DotImageWebFormsDemo
{
	public partial class WebFormsDemoAnnotations : System.Web.UI.Page
	{
        protected static bool _hasPDF = false;

		static WebFormsDemoAnnotations()
		{
            try
            {
                Atalasoft.Imaging.Codec.RegisteredDecoders.Decoders.Add(new Atalasoft.Imaging.Codec.Pdf.PdfDecoder());
                _hasPDF = true;
            }
            catch (Exception)
            {
                _hasPDF = false;
            }
		}

		protected void Page_Load(object sender, System.EventArgs e)
		{
			// Sets up the default annotation collection
			InitializeDefaultAnnotations();

			// Put user code to initialize the page here
			if (!Page.IsPostBack)
			{
				// Done in case the page is refreshed
				WebAnnotationViewer1.ClearAnnotations();
				// Prepare and open the image
				WebThumbnailViewer1.OpenUrl(PrepareImageForOpen("DocCleanMultipage.tif"));
			}
		}

		#region Remote Invokes

		// This method copies the file at the given path to the cache folder
		[RemoteInvokable]
		public String PrepareImageForOpen(String path)
		{
			string orig = Page.MapPath("SampleImages/" + path);
			string url = "AtalaCache/" + Page.Session.SessionID + "_tn.tif";
			string cached = Page.MapPath(url);

            if (path.EndsWith("pdf") && !_hasPDF)
            {
                return "";
            }

			// copy from Images to cache and make writable
			File.Copy(orig, cached, true);
			FileAttributes attr = File.GetAttributes(cached);
			attr &= ~FileAttributes.ReadOnly;
			File.SetAttributes(cached, attr);

			// We need to make sure that there are the same number of layers as pages
			ImageInfo info = RegisteredDecoders.GetImageInfo(cached);
			if (WebAnnotationViewer1.Annotations.Layers.Count != info.FrameCount)
			{
				while (WebAnnotationViewer1.Annotations.Layers.Count < info.FrameCount)
				{
					WebAnnotationViewer1.Annotations.Layers.Add(new LayerAnnotation());
				}

				while (WebAnnotationViewer1.Annotations.Layers.Count > info.FrameCount)
				{
					WebAnnotationViewer1.Annotations.Layers.RemoveAt(WebAnnotationViewer1.Annotations.Layers.Count - 1);
				}

				WebAnnotationViewer1.UpdateAnnotations();
			}

            return url;
		}

		// This method is not used in the current demo, it is here to  
		// demonstrate how to make changes to a Tiff document
		[RemoteInvokable]
		public void Remote_ProcessImage(string commandName)
		{
			// Create a new command based on the user's selection
			this.ApplyCommand(GetCommand(commandName));
		}

		// called by the client when Properties... is chosen from the context menu
		[RemoteInvokable]
		public String PopupAnnoDialog(int layerIndex, int index)
		{
			AnnotationUI anno = WebAnnotationViewer1.Annotations.GetAnnotation(layerIndex, -1, index);
			AnnotationData d = anno.Data;

			String outlineColor = "";
			int outlineWidth = 0;
			String fillColor = "";

			// Outline and Fill properties are not in the base AnnotationData
			// object, so we need to use PropertyInfo to get them on all annotation types
			PropertyInfo outlineProp = d.GetType().GetProperty("Outline");
			PropertyInfo fillProp = d.GetType().GetProperty("Fill");

			if (outlineProp != null)
			{
				AnnotationPen outline = (AnnotationPen)(outlineProp.GetValue(d, null));
				if (outline != null)
				{
					outlineColor = FormatColor(outline.Color);
					outlineWidth = (int)outline.Width;
				}
			}

			if (fillProp != null)
			{
				AnnotationBrush fill = (AnnotationBrush)(fillProp.GetValue(d, null));
				if (fill != null)
				{
					fillColor = FormatColor(fill.Color);
				}
			}

			return outlineColor + ":" + outlineWidth + ":" + fillColor;
		}

		// called by the client when the Properties... is saved
		[RemoteInvokable]
		public String SaveProps(int layerIndex, int index, String outlineColor, String outlineWidth, String fillColor)
		{
			try 
			{
				AnnotationUI anno = WebAnnotationViewer1.Annotations.GetAnnotation(layerIndex, -1, index);
				AnnotationData d = anno.Data;

				// Outline and Fill properties are not in the base AnnotationData
				// object, so we need to use PropertyInfo to get them on all annotation types
				PropertyInfo outlineProp = d.GetType().GetProperty("Outline");
				PropertyInfo fillProp = d.GetType().GetProperty("Fill");

				if (outlineProp != null)
				{
					AnnotationPen outline = (AnnotationPen)(outlineProp.GetValue(d, null));
					if (outline != null)
					{
						outline.Color = ParseColor(outlineColor);
						outline.Width = ParseInt(outlineWidth);
					}
				}

				if (fillProp != null)
				{
					AnnotationBrush fill = (AnnotationBrush)(fillProp.GetValue(d, null));
					if (fill != null)
					{
						fill.Color = ParseColor(fillColor);
					}
				}

				WebAnnotationViewer1.UpdateAnnotations();
			}
			catch (Exception e) 
			{
				return e.Message;
			}

			return "";
		}

		#endregion
		
		#region Functions

		private void ApplyCommand(ImageCommand command)
		{
			if (command != null)
			{
				AtalaImage temp = WebAnnotationViewer1.Image;

				if (command.InPlaceProcessing)
					command.Apply(temp);
				else
					temp = command.Apply(temp).Image;

				SaveChanges(temp);
			}
		}

		// This method of saving changes only supports Tiff format
		// (both for the original image and the destination image)
		private void SaveChanges(AtalaImage image)
		{
			string url = this.WebAnnotationViewer1.ImageUrl;
			string path = Page.MapPath(url);
			int frame = this.WebAnnotationViewer1.CurrentPage - 1;

			using (Stream fs = new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.Read))
			{
				TiffFile tFile = new TiffFile();
				tFile.Read(fs);
				
				TiffDirectory tImage = new TiffDirectory(image, TiffCompression.Group4FaxEncoding);
				tFile.Images.RemoveAt(frame);
				tFile.Images.Insert(frame, tImage);
				tFile.Save(path + "_tmp");
			}

			File.Delete(path);
			File.Move(path + "_tmp", path);

			this.WebAnnotationViewer1.OpenUrl(url, frame);
		}

		// Method used to define image commands
		private ImageCommand GetCommand(string commandName)
		{
			ImageCommand command = null;

			switch (commandName)
			{
				case "RotateRight":
					command = new RotateCommand(90.0);
					break;
				case "RotateLeft":
					command = new RotateCommand(-90.0);
					break;
			}

			return command;
		}

		private int ParseInt(String s)
		{
			try 
			{
				return Int32.Parse(s);
			}
			catch (Exception) 
			{
				throw new ArgumentException(s + " is not a valid number.");
			}
		}

		// turns a color into its HTML equivalent
		private String FormatColor(Color c)
		{
			PropertyInfo[] colorList = typeof(Color).GetProperties();

			foreach (PropertyInfo color in colorList)
			{
				if (color.PropertyType.Equals(typeof(Color))) 
				{
					Color knownColor = (Color)(color.GetValue(null, null));
					if (c.R == knownColor.R && c.G == knownColor.G && c.B == knownColor.B && c.A == knownColor.A)
					{
						return color.Name;
					}
				}
			}
			if (c.A < 255) 
			{
				return String.Format("#{0:X2}{1}", c.A, ColorTranslator.ToHtml(c).Substring(1));
			}
			return String.Format("#{0}", ColorTranslator.ToHtml(c).Substring(1));
		}

		private Color ParseColor(String s)
		{	
			try 
			{
				return ColorTranslator.FromHtml(s);
			}
			catch (Exception) 
			{
			}
			try 
			{
				return ColorTranslator.FromHtml("#"+s);
			}
			catch (Exception) 
			{
				throw new ArgumentException(s + " is not a valid color.");
			}
		}

		#endregion

		#region Creating Annotations

		// Method called from Page_Load to setup some basic annotations for creation from JavaScript
		private void InitializeDefaultAnnotations()
		{
			EllipseData ellipse = new EllipseData();
			ellipse.Name = "DefaultEllipse";
			ellipse.Outline = new AnnotationPen(Color.Black, 10);
			ellipse.Fill = new AnnotationBrush(Color.Red);
			ellipse.ToolTip = "Double-click to edit";
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(ellipse);

			RectangleData highlighter = new RectangleData();
			highlighter.Name = "DefaultHighlighter";
			highlighter.Outline = new AnnotationPen(Color.Transparent, 1);
			highlighter.Fill = new AnnotationBrush(Color.FromArgb(128, Color.Yellow));
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(highlighter);

			LineData line = new LineData();
			line.Name = "DefaultLine";
			line.Outline = new AnnotationPen(Color.DarkGreen, 6);
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(line);

			LinesData lines = new LinesData();
			lines.Name = "DefaultLines";
			lines.Outline = new AnnotationPen(Color.DarkOrange, 6);
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(lines);

			FreehandData freehand = new FreehandData();
			freehand.Name = "DefaultFreehand";
			freehand.Outline = new AnnotationPen(Color.DarkBlue, 6);
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(freehand);

			PolygonData polygon = new PolygonData();
			polygon.Name = "DefaultPolygon";
			polygon.Outline = new AnnotationPen(Color.MidnightBlue, 10);
			polygon.Fill = new AnnotationBrush(Color.FromArgb(192, Color.MidnightBlue));
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(polygon);

			RectangleData rectangle = new RectangleData();
			rectangle.Name = "DefaultRectangle";
			rectangle.Outline = new AnnotationPen(Color.Sienna, 10);
			rectangle.Fill = new AnnotationBrush(Color.Plum);
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(rectangle);

			RectangleData redaction = new RectangleData();
			redaction.Name = "DefaultRedaction";
			redaction.Fill = new AnnotationBrush(Color.Black);
			redaction.CanMove = false;
			redaction.CanResize = false;
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(redaction);

			TextData sticky = new TextData();
			sticky.Name = "DefaultStickyNote";
			sticky.Fill = new AnnotationBrush(Color.Yellow);
			sticky.Shadow = new AnnotationBrush(Color.FromArgb(128, 0, 0, 0));
			sticky.ShadowMode = ShadowMode.Annotation;
			sticky.ShadowOffset = new PointF(6, 6);
			sticky.Text = "Default Sticky Note:\n double-click to edit note";
			sticky.Font = new AnnotationFont("Verdana", 32f);
			sticky.Font.Bold = true;
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(sticky);

			TextData text = new TextData();
			text.Name = "DefaultTextAnnotation";
			text.Text = "Default Text Annotation:\n double-click to edit";
			text.Font = new AnnotationFont("Arial", 24f);
			text.FontBrush = new AnnotationBrush(Color.Black);
			text.Fill = new AnnotationBrush(Color.Ivory);
			text.Outline = new AnnotationPen(new AnnotationBrush(Color.White), 2);
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(text);

			ReferencedImageData stamp = new ReferencedImageData();
			stamp.Name = "DefaultStamp";
			stamp.FileName = Page.MapPath("SampleImages/draft.png");
			WebAnnotationViewer1.Annotations.DefaultAnnotations.Add(stamp);
		}

		#endregion
	}
}
